/*
 * Super Simple Serial
 *
 * Non-canonical asynchronous serial port communication
 *
 * This is not your common Swiss Army Knife of serial port functions.
 * Rather, it is more like a scalpel, cutting closely to the needs of
 * everyday serial port communication. It does one thing well. It writes
 * data to serial ports, and receives data from serial ports. No muss.
 *
 *	- No hardware handshaking (RTS/CTS)
 *	- No software flow control (XON/XOFF)
 *	- No line handling (CR/LF)
 *	- No input processing; raw data passthrough only
 *	- No permanent blocking; uses data timeouts only
 *	- No data assumptions; supports both ASCII and binary
 *
 * The reading of data is performed on a per message basis, where the length
 * of the message can be unknown. A received message is deemed complete when
 * either a specified number of bytes are received or when 100 msec of time
 * elapses with no new bytes input.
 *
 * If a byte is not received within a timeframe, the message is terminated.
 * This timeout value for the initial message byte is 500 msec, and each
 * subsequent byte is expected within 100 msec. These timeouts work well for
 * baudrates above 1200. Timeouts must be increased for a baudrate of 1200
 * or below.
 *
 * If a message length is not specified then there will be a delay of 100
 * msec after the last byte is received while we wait for possible new
 * data. This means that the minimum length of such messages is 100 msec,
 * and that no more than 10 messages can be received per second. The highest
 * performance, is therefore obtained when the received message length is
 * known, and specified, in advance.
 *
 * Here is a quick little usage example that sends a message and then prints
 * a received message:
 *
 *	int fd = serial_open("/dev/ttyUSB0");
 *	serial_setup(fd, B38400, CS8, STOP_BITS_1, PARITY_NONE);
 *	serial_write(fd, "Hello, world!\n", 14);
 *	serial_read_to_file(fd, stdout, 0);
 *	serial_close(fd);
 *
 * If you connect the TX/RX lines together on your serial port, this example
 * will display 'Hello, world!' on your terminal.
 *
 *
 * Copyright:
 *
 * 	Copyright (c) 2010 by Timothy Wagner. All Rights Reserved.
 *
 */

#include <stdlib.h>
#include <errno.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/time.h>
#include "serial.h"

// Number of bytes read at a time
#define	READ_CHUNK	128

// Timeout for first byte received in milliseconds
#define TIMEOUT_FIRST	500

// Timeout to next byte received in milliseconds, in 100 msec steps
#define	TIMEOUT_NEXT	100

// Elapsed read time in microseconds
static long microsecs;

/*
 * Open bidirectional serial port
 *
 * Parameters:
 *
 *	device		- a string containing the path to the device file.
 *			  For example: /dev/ttyUSB0
 *
 * Returns:
 *
 *	int		- file descriptor for serial device
 *
 */
int serial_open(char *device)
	{
	int fd_serial = open(device, O_RDWR | O_NOCTTY);
	if (fd_serial == -1)
		{
		perror("serial_open: open");
		return -1;
		}

	return fd_serial;
	}

/*
 * Close serial port
 *
 * Parameters:
 *
 *	fd_serial		- file descriptor for serial device
 *
 * Returns:
 *
 *	int			- 0 on OK and -1 on failure
 *
 */
int serial_close(int fd_serial)
	{
	int	rc;

	rc = close(fd_serial);
	if (rc == -1)
		perror("serial_close: close");
	return rc;
	}

/*
 * Configure serial port
 *
 * Parameters:
 *
 *	fd_serial		- file descriptor for serial device
 *
 *	baud_rate		- select one of the following baud rates
 *
 *					B50
 *					B75
 *					B110
 *					B134
 *					B150
 *					B200
 *					B300
 *					B600
 *					B1200
 *					B1800
 *					B2400
 *					B4800
 *					B9600
 *					B19200
 *					B38400
 *					B57600
 *					B76800
 *					B115200
 *
 *	data_bits		- select one of the following data widths
 *
 *					CS5
 *					CS6
 *					CS7
 *					CS8
 *
 *	stop_bits		- select one of the following stop bits
 *
 *					STOP_BITS_1
 *					STOP_BITS_2
 *
 *	parity			- selct one of the following parity bits
 *
 *					PARITY_NONE
 *					PARITY_ODD
 *					PARITY_EVEN
 *
 * Returns:
 *
 *	int			- 0 on Ok, -1 on FAIL
 *
 */
int serial_setup(int fd_serial, int baud_rate,
	int data_bits, int stop_bits, int parity)
	{
	struct termios options;

	memset(&options, 0, sizeof options);
        if (tcgetattr(fd_serial, &options) != 0)
		{
		perror("serial_setup: tcgetattr");
		return -1;
		}

	// Set baud rate
	cfsetospeed(&options, baud_rate);
	cfsetispeed(&options, baud_rate);

	// Set number of data bits
	options.c_cflag &= ~CSIZE;
	options.c_cflag |= data_bits;

	// Select 2 stop bits
	if (stop_bits == STOP_BITS_2)
		options.c_cflag |= CSTOPB;

	// Select 1 stop bit
	else
		options.c_cflag &= ~CSTOPB;

	// Select odd parity
	if (parity == PARITY_ODD)
		{
		options.c_cflag |= PARENB;
		options.c_cflag |= PARODD;
		}

	// Select even parity
	else if (parity == PARITY_EVEN)
		{
		options.c_cflag |= PARENB;
		options.c_cflag &= ~PARODD;
		}

	// Select no parity
	else
		options.c_cflag &= ~PARENB;

	// By default disable all delays and blocking
	options.c_oflag = 0;
	options.c_cc[VMIN] = 0;
	options.c_cc[VTIME] = 0;

	// Select raw data input
	options.c_lflag &= ~(ICANON | ECHO | ECHOE | ISIG);

	// Select raw data output
	options.c_oflag &= ~OPOST;

	// Disable xon/xoff ctrl
	options.c_iflag &= ~(IXON | IXOFF | IXANY);

	// Disable hardware flow control
	options.c_cflag &= ~CRTSCTS;

	// Disable break processing
	options.c_iflag &= ~IGNBRK;

	// Ignore modem controls, enable reading
	options.c_cflag |= (CLOCAL | CREAD);

	if (tcsetattr(fd_serial, TCSANOW, &options) != 0)
		{
		perror("serial_setup: tcsetattr");
		return -1;
		}

	return 0;
	}

/*
 * ---- Internal Use Only ----
 *
 * Choose data available blocking mode
 *
 * There are four blocking modes as determined by the values of the count
 * and timeout parameters.
 *
 *	Disable Blocking	- set both count and timeout to zero.
 *				  Reads available data and returns immediately.
 *
 *	Initial Timeout		- set count to zero and timeout to msecs.
 *				  Waits for timeout for first byte,
 *				  then return all available data.
 *
 *	Intercharacter Timeout	- set count and timeout to non-zero.
 *				  Block forever until first byte available,
 *				  then wait for each byte until count is
 *				  satisfied or time elapses.
 *
 *	Block Forever		- set timeout to zero and count to read size.
 *				  Waits forever until count is satisfied.
 *
 * Parameters:
 *
 *	fd_serial		- file descriptor for serial device
 *
 *	count			- number of bytes to satisfy read
 *
 *	timeout			- timeout on wait for byte
 *
 *
 * Returns:
 *
 *	int			- 0 on Ok, -1 on FAIL
 *
 */
static int serial_blocking(int fd_serial, int count, int timeout)
	{
	struct termios options;

	memset(&options, 0, sizeof options);
	if (tcgetattr(fd_serial, &options) != 0)
		{
		perror("serial_blocking: tcgetattr");
		return -1;
		}

	options.c_cc[VMIN] = count;
	options.c_cc[VTIME] = timeout;

	if (tcsetattr(fd_serial, TCSANOW, &options) != 0)
		perror("serial_blocking: tcsetattr");

	return 0;
	}

/*
 * ---- Internal Use Only ----
 *
 * Read block of serial data with timeouts
 *
 * Attempt to read a specified number of bytes, not to exceed the value of
 * READ_CHUNK. If no value is given then the number of bytes read will be
 * specified by READ_CHUNK. READ_CHUNK is defaulted to 128 and cannot
 * exceed 255 in size.
 *
 * Will read both ASCII and binary data. Completed data buffer is terminated
 * with a NULL (0) just in case the data is ASCII.
 *
 * Failure to read the specified number of bytes will be because of timeout.
 * The value of to_first, in milliseconds, is the wait time for the
 * first byte to arrive. The value of to_next is the wait time for
 * each additional byte to arrive. Though to_next is specified in
 * milliseconds, it must be in multiples of 100 msec. Valid values are 100,
 * 200, 300, 400, etc.
 *
 * Parameters:
 *
 *	fd_serial		- file descriptor for serial device
 *
 *	buffer			- pointer to buffer of at least READ_CHUNK
 *				  plus one in size.
 *
 *	count			- number of bytes to read. If set to zero
 *				  then READ_CHUNK is the value for count.
 *
 *	to_first		- milliseconds timeout on first byte
 *
 *	to_next			- milliseconds timeout on subsequent bytes.
 *				  Must be multiples of 100 msec.
 *
 * Returns:
 *
 *	int			- count of bytes read
 *
 */
static unsigned int serial_read_chunk(int fd_serial, char *buffer, int count,
	int to_first, int to_next)
	{
	struct timeval	tv;
	fd_set		read_set;
	unsigned int	read_size;
	int		rc;

	if (count <= 0 || count > READ_CHUNK)
		count = READ_CHUNK;

	// Initialize the set
	FD_ZERO(&read_set);
	FD_SET(fd_serial, &read_set);
   
	// Initialize time out struct
	tv.tv_sec = 0;
	tv.tv_usec = to_first * 1000;

	// select()
	rc = select(fd_serial + 1, &read_set, NULL, NULL, &tv);

	// Check status
	if (rc <= 0 || FD_ISSET(fd_serial, &read_set) == 0)
		return 0;

	serial_blocking(fd_serial, count, to_next / 100);
	read_size = read(fd_serial, buffer, count);
	serial_blocking(fd_serial, 0, 0);
	if (read_size < 0)
		read_size = 0;
	buffer[read_size] = 0;

	return read_size;
	}

/*
 * Read all available serial data to dynamic buffer
 *
 * malloc and realloc are used to create a buffer for storage of serial
 * data. The user is responsibile for calling free() to prevent memory leaks.
 *
 * If the system runs out of memory and the buffer cannot be extended to
 * hold additional data, then the data accumulated so far will be returned.
 * In such case the additional data will be likely forfeited.
 *
 * Parameters:
 *
 *	fd_serial		- file descriptor for serial device
 *
 *	total_size		- pointer to count of bytes read upon return
 *
 *	request_size		- number of bytes to read. A value of
 *				  zero will read until timeout occurs.
 *
 * Returns:
 *
 *	char *			- pointer to data buffer created with malloc.
 *				  NULL is return if initial memory could not
 *				  be allocated.
 *
 */
char *serial_read_to_malloc(int fd_serial, unsigned long *total_size,
	long request_size)
	{
	struct timeval	start, end;
	unsigned int	chunk_size;
	int		read_size;
	int		countit;
	int		to_first;
	int		to_next;
	char		*buffer;
	char		*new_buffer;

	gettimeofday(&start, NULL);
	*total_size = 0;
	buffer = malloc(READ_CHUNK + 1);
	if (buffer == NULL)
		{
		perror("serial_read_to_malloc: malloc");
		microsecs = 0;
		return (NULL);
		}
	buffer[0] = 0;
	to_first = TIMEOUT_FIRST;
	to_next = TIMEOUT_NEXT;
	chunk_size = READ_CHUNK;
	countit = request_size > 0 ? 1 : 0;
	while (1)
		{
		if (countit)
			{
			if (request_size < READ_CHUNK)
				chunk_size = request_size;
			request_size -= chunk_size;
			}
		read_size =  serial_read_chunk(fd_serial, &buffer[*total_size],
			chunk_size, to_first, to_next);
		to_first = TIMEOUT_NEXT;
		*total_size += read_size;
		if (read_size < chunk_size || (countit && request_size <= 0))
			break;
		new_buffer = realloc(buffer, *total_size + READ_CHUNK + 1);
		if (new_buffer == NULL)
			{
			perror("serial_read_to_malloc: realloc");
			break;
			}
		buffer = new_buffer;
		}

	gettimeofday(&end, NULL);
	microsecs = (end.tv_sec - start.tv_sec) * 1000000 +
		end.tv_usec - start.tv_usec;

	return buffer;
	}

/*
 * Read all available serial data to file
 *
 * Most useful for directing all read serial data to stdout.
 *
 * Parameters:
 *
 *	fd_serial		- file descriptor for serial device
 *
 *	output_file		- pointer of type FILE for use with
 *				  standard C library functions.
 *
 *	request_size		- number of bytes to read. A value of
 *				  zero will read until timeout occurs.
 *
 *
 * Returns:
 *
 *	unsigned long		- count of bytes read
 *
 */
unsigned long serial_read_to_file(int fd_serial, FILE *output_file,
	long request_size)
	{
	struct timeval	start, end;
	unsigned long	total_size;
	unsigned int	read_size;
	unsigned int	chunk_size;
	int		countit;
	int		to_first;
	int		to_next;
	char		buffer[READ_CHUNK + 1];

	gettimeofday(&start, NULL);
	to_first = TIMEOUT_FIRST;
	to_next = TIMEOUT_NEXT;
	total_size = 0;
	chunk_size = READ_CHUNK;
	countit = request_size > 0 ? 1 : 0;
	while (1)
		{
		if (countit)
			{
			if (request_size < READ_CHUNK)
				chunk_size = request_size;
			request_size -= chunk_size;
			}
		read_size = serial_read_chunk(fd_serial, buffer, chunk_size,
			to_first, to_next);
		to_first = TIMEOUT_NEXT;
		total_size += read_size;
		fwrite(buffer, 1, read_size, output_file);
		if (read_size < chunk_size || (countit && request_size <= 0))
			break;
		}

	gettimeofday(&end, NULL);
	microsecs = (end.tv_sec - start.tv_sec) * 1000000 +
		end.tv_usec - start.tv_usec;

	return total_size;
	}

/*
 * Write to serial port
 *
 * Use as a substitute for the standard write() function with same
 * calling parameters.
 *
 * Parameters:
 *
 *	fd_serial		- file descriptor for serial device
 *	
 *	buffer			- pointer to buffer containing bytes to
 *				  send out serial port
 *
 *	count			- number of bytes to be written
 *
 * Returns:
 *
 *	int			- number of bytes written or -1 on failure
 *
 */
int serial_write(int fd_serial, void *buffer, int count)
	{
	int	bytes_written;

	bytes_written = write(fd_serial, buffer, count);
	if (bytes_written == -1)
		perror("serial_write: write");
	return bytes_written;
	}

/*
 * Serial read operations timer
 *
 * The elapsed time of each call to read serial data is recorded.
 *
 * Returns:
 *
 *	long			- microseconds run time of last serial
 *				  read request.
 *
 */
long serial_elapsed_microsecs()
	{
	return microsecs;
	}
